// PortMessage.h starts

// Class for sending messages through ports to Experimenter or Rat
// Changes
//		CGP; 11/2/00; See .h file.

#include <OS.h>
#include <Message.h>
#include "PortMessage.h"
#include "Debug.h"

PortMessage::PortMessage(char *name)
{
	if (name == NULL) {
		name = "";
	}
	
	port = create_port(20L, name);
	if (port < 0) { // not sure what port numbers constitute a bad port...
		DebugBoot('e', "PortMessage::PortMessage: Bad port: Name= ", name);
	}
	
	peekBuffer = NULL; // nothing in peek buffer
}

PortMessage::~PortMessage()
{
}

BMessage *PortMessage::PeekMessage()
{
	if (peekBuffer != NULL) { // something in peek buffer, return that
		return peekBuffer;
	} else { // nothing in peek buffer, get new message
		peekBuffer = GetMessage();
		return peekBuffer;
	}
}

bool PortMessage::IsMessageReady()
{
	if (peekBuffer == NULL) { // if nothing in peek buffer, check for new incoming messages
		// this will tell us if any message is ready at the port without reading the port
		return (port_count(port) != 0);
	} else {
		return true;
	}
}

void PortMessage::Flush()
{
	while (IsMessageReady()) {
		delete GetMessage();
	}
}

BMessage *PortMessage::GetMessage()
{
	int32 msg;
	char *buffer = NULL;
	ssize_t bufferSize = 0;
	BMessage *bmsg;
	status_t stat;
	
	if (peekBuffer == NULL) { // nothing in peek buffer, read from port
		// Get the size of the message waiting for us. Blocks if  no message
		// is ready.
		bufferSize = port_buffer_size(port);
		
		if (bufferSize < 0) {
			// assume error codes are values of size < 0
			DebugBoot('e', "PortMessage::GetMessage: Error: Size of port buffer is negative");
			return NULL;
		} else if (bufferSize > 0) {
			buffer = new char[bufferSize];
			if (buffer == NULL) {
				DebugBoot('e', "PortMessage::GetMessage: Error: Cannot allocate buffer");
				return NULL;
			}
		}
		
		// read_port returns the number of bytes written into the buffer.
		// Only an error if less than zero.
		if (read_port(port, &msg, buffer, bufferSize) < 0) {
			DebugBoot('e', "PortMessage::GetMessage: Error reading from port");
			delete [] buffer;
			return NULL;
		}
		
		// So, have got a message from the port
		bmsg = new BMessage();
		
		bmsg->what = msg; // set the code
		stat = bmsg->Unflatten(buffer);
		delete [] buffer;
		
		if (stat != B_OK) {
			DebugBoot('e', "PortMessage::GetMessage: Error: Cannot unflatten message");
			delete bmsg;
			return NULL; 
		}
	} else { // something in peek buffer, so return that
		DebugBoot('n', "PortMessage::GetMessage: peekBuffer is not null");
		bmsg = peekBuffer;
		peekBuffer = NULL;
	}
	
	return bmsg;
}

void PortMessage::SendMessage(BMessage *bmsg)
{
	char *buffer = NULL;
	ssize_t bufferSize = 0;
	status_t stat;
	//ssize_t numBytes = 0;
	
	bufferSize = bmsg->FlattenedSize();
	if (bufferSize < 0) {
		DebugBoot('e', "PortMessage::SendMessage: Error: Could not get flattened size");
		return;
	}
	
	if (bufferSize == 0) {
		DebugBoot('n', "PortMessage::SendMessage: FlattenedSize() is zero");
		return;
	}	
	
	if (bufferSize > 0) {
		buffer = new char[bufferSize];
		if (buffer == NULL) {
			DebugBoot('e', "PortMessage::SendMessage: Error: Could not allocate buffer");
			return;
		}
	}
	
	// CGP, 11/2/00: There appears to be a problem with the Flatten documentation.
	// The documentation indicates that the number of bytes written using Flatten()
	// should be returned in the second argument to flatten, hence, the second argument
	// should be a pointer. But, compile errors occur when you pass a pointer
	// as a second argument. The Flatten() documentation also indicates that
	// you can omit the second argument (in char * form) as it has a default, but omitting
	// the second argument also generates a compiler error.
	//stat = bmsg->Flatten(buffer, numBytes);
	stat = bmsg->Flatten(buffer, bufferSize);
	if (stat != B_OK) {
		DebugBoot('e', "PortMessage::SendMessage: Error flattening message");
		delete [] buffer;
		return;
	}
	
	//if (numBytes == 0) {
	//	DebugBoot('e', "PortMessage::SendMessage: Error: numBytes is zero");
	//}
		
	stat = write_port(port, bmsg->what, (void *) buffer, bufferSize);
	delete [] buffer;
	
	if (stat != B_OK) {
		DebugBoot('e', "PortMessage::SendMessage: Error writing message to port");
	}
}

void PortMessage::SendMessage(int msg_code)
{
	BMessage bmsg((uint32) msg_code);
	
	SendMessage(&bmsg);
}

// PortMessage.h ends